<?php

use CoinAccepted\Payment\CoinAccepted;
use CoinAccepted\Payment\Helper;
use CoinAccepted\Payment\Payment;
use CoinAccepted\Payment\Util;

class WC_Gateway_CoinAccepted extends WC_Payment_Gateway
{
	const PAYMENT_METHOD = 'ca_cryptocurrencies';
    const TRAVEL_RULE_URL = 'https://coinaccepted.com/en/helpdesk/bitcoin-cryptocurrencies/travel-rule';

    public function __construct()
	{

		$this->id = self::PAYMENT_METHOD;
		$this->icon = self::get_image_src('logo.svg');
		$this->has_fields = false;
		$this->description = __('Cryptocurrency payments.', 'coinaccepted');
		$this->title = $this->get_option('coinaccepted_payment_title')
			?: 'cryptocurriencies';

        $this->supports = array('products');

		// Admin
		$this->method_title = __('CoinAccepted', 'coinaccepted');
		$this->method_description = __('Cryptocurrency payments.', 'coinaccepted');

		$this->init_form_fields();
		$this->init_settings();

		//Actions
		add_action('woocommerce_update_options_payment_gateways_' . $this->id, [
			$this,
			'process_admin_options',
		]);
		add_action('woocommerce_receipt_' . $this->id, [
			$this,
			'receipt_page',
		]);
		add_action('woocommerce_api_' . strtolower(get_class($this)), [
			$this,
			'coinaccepted_notification',
		]);

		//Filters
		add_filter('woocommerce_available_payment_gateways', [
			$this,
			'check_coinaccepted_available_payment_gateways',
		]);

        //Styles
        add_action('wp_enqueue_scripts', [$this, 'load_custom_scripts']);
	}

	private static function get_notification_url(): string
	{
		return add_query_arg('wc-api', 'wc_gateway_ca_cryptocurrencies', home_url('/'));
	}

	private static function get_language(): string
	{

		return get_locale() === 'pl_PL'
			? 'pl'
			: 'en';
	}

	public function get_image_src(string $file): string
	{
		return WOOCOMMERCE_COINACCEPTED_PLUGIN_URL . 'resources/images/' . $file;
	}

	public function init_form_fields(): void
	{
		$this->form_fields = [
			'coinaccepted_enabled' => [
				'title'   => __('Enable / Disable', 'coinaccepted'),
				'type'    => 'checkbox',
				'label'   => __('Enable plugin', 'coinaccepted'),
				'default' => 'yes',
			],

			'coinaccepted_private_key'   => [
				'title'   => __('Private key', 'coinaccepted'),
				'type'    => 'text',
				'default' => '',
			],
			'coinaccepted_public_key'    => [
				'title'   => __('Public key', 'coinaccepted'),
				'type'    => 'text',
				'default' => '',
			],
			'coinaccepted_payment_title' => [
				'title'       => __('Payment title', 'coinaccepted'),
				'type'        => 'text',
				'placeholder' => $this->title,
			],
			'coinaccepted_notification'  => [
				'title'       => __('Notification URL', 'coinaccepted') . ': ',
				'type'        => 'title',
				'description' => self::get_notification_url(),
			],
		];
	}

	public function process_payment($order_id): array
	{
		$order = new WC_Order($order_id);

		return [
			'result'   => 'success',
			'redirect' => $order->get_checkout_payment_url(true),
		];
	}

	public function receipt_page(string $order_id): void
	{

		header("Cache-Control: no-cache, no-store, must-revalidate");

		if(empty(WC()->cart->cart_contents)) {
			wp_redirect(get_home_url());

			return;
		}

		global $wp_query;

		$payment = new Payment(
			$this->get_option('coinaccepted_private_key'),
			$this->get_option('coinaccepted_public_key')
		);

		$order = new WC_Order($order_id);

		try {
			$paymentResult = $payment->buildOrder(
				$order->get_currency(),
				$order_id,
				(float) $order->get_total(),
				$order->get_billing_email(),
				self::get_notification_url(),
				$this->get_return_url($order),
				$this->get_return_url($order),
			);
		} catch(Exception $e) {

			$lang = self::get_language();

			$logger = wc_get_logger();
			$logger->error(Helper::getErrorMessage($e->getMessage(), $lang) . ' ' . current_filter() . PHP_EOL);

			$wp_query->query_vars['error_message'] = Helper::getErrorMessage(Helper::ERROR_MESSAGE_FOR_CUSTOMER, $lang);
			$wp_query->query_vars['error'] = true;

			load_template(__DIR__ . '/templates/redirect.php', false);

			return;
		}

		$wp_query->query_vars['error'] = false;

		$wp_query->query_vars['paymentResult'] = $paymentResult;

		// Remove cart
		WC()->cart->empty_cart();

		// region Send email
		WC()->mailer();
		$mail = new WC_Email_New_Order();
		$mail->trigger($order_id);

		$mail2 = new WC_Email_Customer_Processing_Order();
		$mail2->trigger($order_id);
		// endregion

		load_template(__DIR__ . '/templates/redirect.php', false);
	}

	public function coinaccepted_notification(): void
	{
        $logger = wc_get_logger();

		if(!($payloadDecoded = CoinAccepted::validateNotification())) {
            $logger->error('Notification validation error' . PHP_EOL);
            CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

			return;
		}

		$orderId = $payloadDecoded['orderId'];

		if(!($order = wc_get_order($payloadDecoded['orderId']))) {
            $logger->error(sprintf('Invalid orderId: %s', $orderId) . PHP_EOL);
            CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

			return;
		}

		$order_status = $order->get_status();

		if(($order->post->post_type !== 'shop_order' && $order->post->post_type !== 'shop_order_placehold' ) || ($order_status !== 'pending' && $order_status !== 'failed')) {
            $logger->error(sprintf('Invalid order status: %s and post_type: %s on orderId: %s', $order_status, $order->post->post_type, $orderId) . PHP_EOL);
            CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

			return;
		}

		$transactionApi = CoinAccepted::checkTransaction(
			$payloadDecoded['paymentId'],
			$this->get_option('coinaccepted_private_key'),
			$this->get_option('coinaccepted_public_key')
		);
		
		if(!$transactionApi) {
            $logger->error(sprintf('REST response validation error on orderId: %s', $orderId) . PHP_EOL);
            CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

			return;
		}

		$paidAmount = $transactionApi['data']['amountInDestinationCurrency'];
		if($paidAmount < (float) $order->get_total()) {
            $logger->error(sprintf('Invalid price on orderId: %s, expected: %s, received: %s', $orderId, $order->get_total(), $paidAmount) . PHP_EOL);
            CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

			return;
		}

		switch($transactionApi['data']['status']) {
            case Util::TRANSACTION_STATUS_COMPLETED:
				wc_reduce_stock_levels($order->get_id());
				$status = 'processing';
				break;
            case Util::TRANSACTION_STATUS_ACTION_REQUIRED:
				$order->add_order_note(
					\sprintf(
						__('<b>Coin Accepted</b>: Payer action is required. Need to include information needed for <a target="_blank" href="%1s">Travel Rule</a> message: <a target="_blank" href="%2s">%2s</a>.', 'coinaccepted'),
                        self::TRAVEL_RULE_URL,
                        $payloadDecoded['travelRuleFormUrl'] ?? ''
					)
				);
				CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_200);
				return;
			default:
                $logger->error(sprintf('REST response on orderId: %s with unexpected status: %s', $orderId, $transactionApi['data']['status']) . PHP_EOL);
                CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

				return;
		}

		$order->update_status($status);
        CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_200);
		exit;
	}

	public function check_coinaccepted_available_payment_gateways(array $gateways): array
	{
		if(!$this->is_available()) {
			unset($gateways['ca_cryptocurrencies']);
		}

		return $gateways;
	}

    public function is_available(): bool {
        return Util::canUseForCurrency(get_woocommerce_currency())
            && !empty($this->get_option('coinaccepted_public_key'))
            && !empty($this->get_option('coinaccepted_private_key'));
    }

    public function load_custom_scripts(): void
    {
        wp_enqueue_style('ca_plugin_css', WOOCOMMERCE_COINACCEPTED_PLUGIN_URL . 'resources/css/payments.css');
    }
}
