<?php

use CoinAccepted\Payment\Util;

include_once(_PS_MODULE_DIR_ . 'coinaccepted/libraries/payment-core/vendor/autoload.php');
include_once(_PS_MODULE_DIR_ . 'coinaccepted/models/CoinAcceptedModel.php');

if(!defined('_PS_VERSION_')) {
    exit;
}

/**
 * @property string name
 * @property string tab
 * @property string version
 * @property string author
 * @property int    need_instance
 * @property bool   currencies
 * @property string currencies_mode
 * @property array  ps_versions_compliancy
 * @property string displayName
 * @property string description
 * @property string confirm_uninstall
 * @property int    is_eu_compatible
 */
class CoinAccepted extends PaymentModule
{
    const PAYMENT_COINACCEPTED_NEW_STATUS = 'PAYMENT_COINACCEPTED_NEW_STATUS';
    const PAYMENT_COINACCEPTED_ACTION_REQUIRED_STATUS = 'PAYMENT_COINACCEPTED_ACTION_REQUIRED_STATUS';

    const ORDER_STATE_TRANSLATIONS = [
        self::PAYMENT_COINACCEPTED_NEW_STATUS => [
            'en' => 'Payment CoinAccepted: awaiting for confirmation',
            'pl' => 'Płatność CoinAccepted: oczekuje na potwierdzenie',
        ],
        self::PAYMENT_COINACCEPTED_ACTION_REQUIRED_STATUS => [
            'en' => 'Payment CoinAccepted: action required',
            'pl' => 'Płatność CoinAccepted: wymagane działanie',
        ]
    ];

    public function __construct()
    {
        $this->name = 'coinaccepted';
        $this->tab = 'payments_gateways';
        $this->version = '1.2.0';
        $this->author = 'CoinAccepted';
        $this->need_instance = 1;
        $this->currencies = true;
        $this->currencies_mode = 'checkbox';
        $this->ps_versions_compliancy = [
            'min' => '1.7',
            'max' => _PS_VERSION_,
        ];
        $this->is_eu_compatible = 1;

        parent::__construct();

        $this->displayName = $this->l('CoinAccepted');
        $this->description = $this->l('Cryptocurrency payments');
        $this->confirm_uninstall = $this->l('Are you sure you want to uninstall CoinAccepted module?');
    }

    /**
     * @param string $name
     * @param string $type
     *
     * @return string
     */
    public static function buildTemplatePath($name, $type)
    {
        return 'module:coinaccepted/views/templates/' . $type . '/' . $name . '.tpl';
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function install(): bool
    {
        if(!(
            parent::install()
            && $this->registerHook('paymentOptions')
            && $this->registerHook('paymentReturn')
            && $this->registerHook('displayBackOfficeHeader')
            && $this->registerHook('displayHeader')
            && $this->registerHook('displayAdminOrderMain')
            && $this->registerHook('displayAdminOrderLeft')
            && Configuration::updateValue('COINACCEPTED_PUBLIC_KEY', '')
            && Configuration::updateValue('COINACCEPTED_PRIVATE_KEY', '')
            && Configuration::updateValue('COINACCEPTED_GA', 0)
        )) {

            return false;
        }

        $this->createModuleTable();

        $this->addOrderState(self::PAYMENT_COINACCEPTED_NEW_STATUS);
        $this->addOrderState(self::PAYMENT_COINACCEPTED_ACTION_REQUIRED_STATUS);

        return empty($this->_errors);
    }

    public function uninstall(): bool
    {
        return Configuration::deleteByName('COINACCEPTED_PUBLIC_KEY')
            && Configuration::deleteByName('COINACCEPTED_PRIVATE_KEY')
            && Configuration::deleteByName('COINACCEPTED_GA')
            && parent::uninstall();
    }

    /**
     * Display configuration form.
     *
     * @return mixed
     */
    public function getContent()
    {
        $msg = '';

        if(Tools::isSubmit('submitCoinAccepted')) {
            if($this->saveConfiguration()) {
                $msg = [
                    'type'    => 'success',
                    'message' => $this->l('Configuration updated successfully'),
                ];
            } else {
                $msg = [
                    'type'    => 'error',
                    'message' => $this->l('There was an error saving your configuration'),
                ];
            }
        }

		$this->context->smarty->assign([
			'coinaccepted_form'                  => './index.php?tab=AdminModules&configure=coinaccepted&token=' . Tools::getAdminTokenLite('AdminModules') . '&tab_module=' . $this->tab . '&module_name=coinaccepted',
			'coinaccepted_enabled'               => Configuration::get('COINACCEPTED_ENABLED'),
			'coinaccepted_public_key'            => Configuration::get('COINACCEPTED_PUBLIC_KEY'),
			'coinaccepted_private_key'           => Configuration::get('COINACCEPTED_PRIVATE_KEY'),
			'coinaccepted_notification_url'      => $this->context->link->getModuleLink('coinaccepted', 'notification'),
			'coinaccepted_msg'                   => $msg,
			'coinaccepted_payment_title'         => Configuration::get('COINACCEPTED_PAYMENT_TITLE'),
			'coinaccepted_payment_title_default' => $this->getPaymentTitleDefault(),
			'coinaccepted_ga'                    => Configuration::get('COINACCEPTED_GA'),
		]);

        return $this->fetchTemplate('admin.tpl');
    }

    /**
     * @param string $name
     *
     * @return mixed
     */
    public function fetchTemplate($name)
    {
        return $this->display(__FILE__, $name);
    }

    /**
     * Hook that run on mobile version of PrestaShop.
     */
    public function hookDisplayMobileHeader()
    {
        $this->hookHeader();
    }

    /**
     * Hook that adds css for old versions of PrestaShop.
     */
    public function hookHeader()
    {
        $this->context->controller->addCSS(($this->_path) . 'assets/css/coinaccepted-legacy-front.min.css', 'all');
    }

    /**
     * Hook that adds css and js in admin panel. For newer version of PrestaShop.
     */
    public function hookDisplayBackOfficeHeader()
    {
        $this->context->controller->addJquery();
        $this->context->controller->addCSS($this->_path . 'assets/css/coinaccepted-admin.min.css', 'all');
        $this->context->controller->addCSS($this->_path . 'assets/css/font-awesome.min.css', 'all');
    }

    /**
     * Hook that adds css and js in admin panel.
     */
    public function hookBackOfficeFooter()
    {
        echo '<link type="text/css" rel="stylesheet" href="' . $this->_path . 'assets/css/coinaccepted-admin.min.css">';
        echo '<link type="text/css" rel="stylesheet" href="' . $this->_path . 'assets/font-awesome.min.css">';

        echo '<script src="' . $this->_path . 'assets/js/coinaccepted-legacy-admin.js"></script>';
    }

    /**
     * @param array $params
     *
     * @return array|bool
     * @throws SmartyException
     * @throws Exception
     */
    public function hookPaymentOptions($params)
    {

        $currency = Currency::getCurrency($params['cart']->id_currency);

        if(
            !Configuration::get('COINACCEPTED_PRIVATE_KEY')
            || !Configuration::get('COINACCEPTED_PUBLIC_KEY')
            || !$this->active
            || !Util::canUseForCurrency($currency['iso_code'])
        ) {
            return false;
        }

        $link = $this->context->link->getModuleLink('coinaccepted', 'payment');

		$this->context->smarty->assign([
			'coinaccepted_logo' => Media::getMediaPath(_PS_MODULE_DIR_ . $this->name . '/assets/img/logo.svg'),
		]);

        $paymentOptions = [];
        $paymentOptionCoinAccepted = new PrestaShop\PrestaShop\Core\Payment\PaymentOption();
        $paymentOptions[] = $paymentOptionCoinAccepted->setCallToActionText(
            Configuration::get('COINACCEPTED_PAYMENT_TITLE') ?: $this->getPaymentTitleDefault())
            ->setModuleName($this->name)
            ->setAction($link)
            ->setAdditionalInformation($this->context->smarty->fetch('module:coinaccepted/views/templates/hook/payment_coinaccepted.tpl'));

        return $paymentOptions;
    }

    /**
     * @return mixed
     */
    public function hookPaymentReturn()
    {
        return $this->display(__FILE__, 'tpl/success.tpl');
    }

    public function hookDisplayAdminOrderMain(array $params)
    {
        $order = new Order($params['id_order']);
        $model = new CoinAcceptedModel($order->id);

        $this->context->smarty->assign(
            [
                'model' => Validate::isLoadedObject($model) ? $model : null,
            ]
        );

        return $this->display(__FILE__, 'views/templates/admin/payment.tpl');
    }

    public function hookDisplayAdminOrderLeft(array $params)
    {
        return $this->hookDisplayAdminOrderMain($params);
    }

    private function getPaymentTitleDefault(): string
    {
        return $this->l('Cryptocurrency payments');
    }

    /**
     * @return bool
     */
    private function saveConfiguration(): bool
    {

        Configuration::updateValue('COINACCEPTED_PUBLIC_KEY', pSQL(Tools::getValue('coinaccepted_public_key')));
        Configuration::updateValue('COINACCEPTED_PRIVATE_KEY', pSQL(Tools::getValue('coinaccepted_private_key')));
        Configuration::updateValue('COINACCEPTED_PAYMENT_TITLE', pSQL(Tools::getValue('coinaccepted_payment_title')));
        Configuration::updateValue('COINACCEPTED_GA', pSQL(Tools::getValue('coinaccepted_ga')));

        return true;
    }

    private function addOrderState(string $stateName): void
    {
        $stateId = Configuration::get($stateName);
        if (
            Validate::isInt($stateId)
            && Validate::isLoadedObject(new OrderState($stateId))
            && false === (bool) (new OrderState($stateId))->deleted
        ) {
            return;
        }

        $orderStateNew = new OrderState();
        $missingLang = true;

        $langs = self::ORDER_STATE_TRANSLATIONS[$stateName];
        foreach ($langs as $lang => $message) {
            $langId = Language::getIdByIso($lang);
            if (isset($langId)) {
                $orderStateNew->name[$langId] = $message;
                $missingLang = false;
            }
        }

        if ($missingLang) {
            $langId = (int)$this->context->language->id;
            $orderStateNew->name[$langId] = $langs['en'];
        }

        $orderStateNew->send_email = false;
        $orderStateNew->invoice = false;
        $orderStateNew->unremovable = false;
        $orderStateNew->color = "#add8e6";
        $orderStateNew->module_name = $this->name;

        if (!$orderStateNew->add()) {
            $this->_errors[] = $this->l('There was an Error installing the module. Cannot add new order state.');

            return;
        }

        if (!Configuration::updateValue($stateName, $orderStateNew->id)) {
            $this->_errors[] = $this->l('There was an Error installing the module. Cannot update new order state.');

            return;
        }

        Tools::copy(_PS_MODULE_DIR_ . $this->name . '/logo.gif', _PS_ORDER_STATE_IMG_DIR_ . $orderStateNew->id . '.gif');
    }

    private function getModuleTemplatePath(): string
    {
        return sprintf('@Modules/%s/views/templates/admin/', $this->name);
    }

    /**
     * @throws Exception
     */
    private function createModuleTable(): void
    {
        $table = addslashes(_DB_PREFIX_ . CoinAcceptedModel::TABLE);
        $sql = '
          CREATE TABLE IF NOT EXISTS `' . $table . '` (
            `order_id` INT NOT NULL PRIMARY KEY,
			`transaction_id` VARCHAR(100) NOT NULL,
			`action_required_link` VARCHAR(255) NOT NULL
		  );';
        $success = Db::getInstance()->Execute($sql);

        if (!$success) {
            throw new Exception('Cannot create module table');
        }
    }

    public function isUsingNewTranslationSystem()
    {
        return true;
    }
}
