<?php

use CoinAccepted\Payment\CoinAccepted;
use CoinAccepted\Payment\Util;

/**
 * @property bool display_column_left
 * @property bool display_header
 * @property bool display_column_right
 * @property bool display_footer
 */
class CoinAcceptedNotificationModuleFrontController extends ModuleFrontController
{

	/**
	 * Class init
	 */
	public function init()
	{
		$this->display_column_left = false;
		$this->display_column_right = false;
		$this->display_footer = false;
		$this->display_header = false;
		parent::init();
	}

	/**
	 * Verify signature, json, create signature and change status if its okay
	 *
	 * @throws Exception
	 */
	public function process(): void
	{
		if(!($payloadDecoded = CoinAccepted::validateNotification())) {
			PrestaShopLogger::addLog(__METHOD__ . ': ' . 'Notification validation error', 1);
            CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

            exit;
		}

		$order = new Order($payloadDecoded['orderId']);

		if(empty($order->id_cart)) {
			PrestaShopLogger::addLog(__METHOD__ . ': ' . 'Notification invalid orderId: ' . $payloadDecoded['orderId'], 1, null, null, null, true);
            CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

			exit;
		}

		if(!($transactionApi = CoinAccepted::checkTransaction($payloadDecoded['paymentId'], Configuration::get('COINACCEPTED_PRIVATE_KEY'), Configuration::get('COINACCEPTED_PUBLIC_KEY')))) {
			PrestaShopLogger::addLog(__METHOD__ . ': ' . 'REST response validation error on orderId: ' . $payloadDecoded['orderId'], 1, null, null, null, true);
            CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

            exit;
		}

		$paidAmount = $transactionApi['data']['amountInDestinationCurrency'];
		if(!$paidAmount || $paidAmount < (float)$order->total_paid) {
			PrestaShopLogger::addLog(__METHOD__ . ': ' . 'Invalid price on orderId: ' . $payloadDecoded['orderId'] . ' Expected: ' . $order->total_paid . ', received: ' . $paidAmount, 1, null, null, null, true);
            CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

            exit;
		}

        $model = new CoinAcceptedModel($order->id);

		switch($transactionApi['data']['status']) {
			case Util::TRANSACTION_STATUS_COMPLETED:
                $model->delete();
				$newState = _PS_OS_PAYMENT_;
				break;
            case Util::TRANSACTION_STATUS_ACTION_REQUIRED:
                $newState = Configuration::get('PAYMENT_COINACCEPTED_ACTION_REQUIRED_STATUS');
                $model->saveLink(
                    $order->id,
                    $payloadDecoded['paymentId'],
                    $payloadDecoded['travelRuleFormUrl'] ?? ''
                );
                break;
			default:
				PrestaShopLogger::addLog(__METHOD__ . ': ' . 'REST response on orderId: ' . $payloadDecoded['orderId'] . ' with unexpected status: ' . $transactionApi['data']['status'], 1, null, null, null, true);
                CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

                exit;
		}

        if((string) $order->current_state === _PS_OS_PAYMENT_) {
            PrestaShopLogger::addLog(__METHOD__ . ': ' . 'Status validation error on orderId:' . $payloadDecoded['orderId'], 1, null, null, null, true);
            CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_404);

            exit;
        }

		$history = new OrderHistory();
		$history->id_order = intval($payloadDecoded['orderId']);

		// Change order state
		$history->changeIdOrderState($newState, intval($payloadDecoded['orderId']));
		$history->addWithemail(true);

        CoinAccepted::responseNotificationHeader(Util::HTTP_CODE_200);
        exit;
	}
}
