<?php

namespace CoinAccepted\Payment;

use Exception;

class CoinAccepted
{
	public static function callApi(
        string $endpoint,
        string $body,
        string $method,
        array $headers,
        bool $returnTransfer = true,
        int $connectTimeout = 10,
        int $timeout = 10
    ): ?string {
		$curl = curl_init($endpoint);

		curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $method);
		curl_setopt($curl, CURLOPT_POSTFIELDS, $body);
		curl_setopt($curl, CURLOPT_RETURNTRANSFER, $returnTransfer);
		curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, $connectTimeout);
		curl_setopt($curl, CURLOPT_TIMEOUT, $timeout);
		curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);

		$response = curl_exec($curl);
		$info = curl_getinfo($curl);
		curl_close($curl);

		if(!isset($info['http_code'])) {
			return null;
		}

		return $response;
	}

	/**
	 * @return bool|array
	 * @throws Exception
	 */
	public static function checkTransaction(string $transactionUuid, string $privateKey, string $publicKey)
	{
		$body = json_encode([]);

		$callResponse = self::callApi(
			Configuration::getServiceUrl(Util::ENVIRONMENT_PRODUCTION) . Configuration::ROUTE_PAYMENTS . '/' . $transactionUuid,
			$body,
			Util::METHOD_GET,
			self::getAuthHeaders($body, $privateKey, $publicKey)
		);

		return json_decode($callResponse, true);
	}

	/**
	 * @return bool|array
	 */
	public static function validateNotification()
	{
		$payload = file_get_contents('php://input', true);

        if(Util::isJson($payload)) {
			return json_decode($payload, true);
		}

		return false;
	}

    /**
     * @throws Exception
     */
	public static function prepareOrderData(
        string $privateKey,
        string $publicKey,
        string $currency,
        string $orderId,
        float $price,
        string $notificationEmail,
        string $notificationsUrl,
        string $successUrl,
        string $failureUrl
    ): PaymentResult {

		$body = json_encode([
			'destinationCurrency' => $currency,
			'orderId'             => $orderId,
			'price'               => $price,
			'successCallbackUrl'  => $successUrl,
			'failureCallbackUrl'  => $failureUrl,
			'notificationsUrl'    => $notificationsUrl,
            'notificationEmail'   => $notificationEmail,
		]);

		$resultCurl = json_decode(self::callApi(
			Util::getServiceUrl(),
			$body,
			Util::METHOD_POST,
			self::getAuthHeaders($body, $privateKey, $publicKey)
		), true);

		if (!isset($resultCurl['status'])) {
			throw new Exception(Helper::ERROR_CONNECT);
		}

		if ($resultCurl['status'] === 'Fail') {
			throw new Exception($resultCurl['errors'][0]['reason']);
		}

        return new PaymentResult(
            $resultCurl['data']['paymentId'],
            $resultCurl['data']['url'],
        );
	}

    public static function responseNotificationHeader(int $httpCode): void
    {
        switch ($httpCode) {
            case Util::HTTP_CODE_200:
                header("HTTP/1.1 200 OK");
                break;
            case Util::HTTP_CODE_404:
                header("HTTP/1.1 404 Not Found");
                break;
            default:
                header("HTTP/1.1 400 Bad Request");
                break;
        }
    }

	private static function getApiHash(string $publicKey, int $timestamp, string $apiSecret, string $body): string
	{
		return bin2hex($hmac = hash_hmac('sha512', $publicKey . $timestamp . $body, $apiSecret, true));
	}

	/**
	 * @throws Exception
	 */
	private static function getAuthHeaders(string $body, string $privateKey, string $publicKey): array
	{

		$timestamp = time();

		return [
			'API-Key:' . $publicKey,
			'API-Hash:' . self::getApiHash($publicKey, $timestamp, $privateKey, $body),
			'operation-id: ' . Util::generateUuidV4(),
			'Request-Timestamp: ' . $timestamp,
			'Content-Type: application/json',
		];
	}
}
