<?php

namespace CoinAccepted\Payment\Controller\Payment;

use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\Result\RawFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Payment\Model\Method\Logger;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\Data\TransactionInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface;
use CoinAccepted\Payment\Util;
use CoinAccepted\Payment\CoinAccepted;
use CoinAccepted\Payment\Block\Adminhtml\PaymentInfo;
use CoinAccepted\Payment\Model\Payment;

class Notification implements CsrfAwareActionInterface, HttpPostActionInterface
{
    private ScopeConfigInterface $scopeConfig;
    private BuilderInterface $builderInterface;
    private OrderInterface $order;
    private OrderRepositoryInterface $orderRepository;
    private Logger $logger;
    private RawFactory $resultRawFactory;

    public function __construct(
		ScopeConfigInterface $scopeConfig,
		BuilderInterface $builderInterface,
        OrderInterface $order,
        OrderRepositoryInterface $orderRepository,
        Logger $logger,
        RawFactory $resultRawFactory
	) {
		$this->scopeConfig = $scopeConfig;
		$this->builderInterface = $builderInterface;
        $this->order = $order;
        $this->orderRepository = $orderRepository;
        $this->logger = $logger;
        $this->resultRawFactory = $resultRawFactory;
    }

    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        return null;
    }

    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }

	public function execute(): ResultInterface
	{
        $response = $this->resultRawFactory->create();

        if (!($payloadDecoded = CoinAccepted::validateNotification())) {
            $this->logger->debug(['message' => 'Notification validation error.']);

            return $response->setHttpResponseCode(Util::HTTP_CODE_404);
		}

        $order = $this->order->loadByIncrementId($payloadDecoded['orderId']);
        if ($order->getEntityId() === null) {
            $order = $this->orderRepository->get($payloadDecoded['orderId']);
        }

		if ($order->getEntityId() === null) {
            $this->logger->debug(['message' => 'Notification invalid orderId. Order not found.', 'order_id' => $order->getEntityId()]);

            return $response->setHttpResponseCode(Util::HTTP_CODE_404);
		}

		if ($order->getState() === Order::STATE_PROCESSING) {
            $this->logger->debug(['message' => 'Order state does not match. Expected different state than PROCESSING.', 'order_id' => $order->getEntityId()]);

            return $response->setHttpResponseCode(Util::HTTP_CODE_404);
		}

		if (!($transactionApi = CoinAccepted::checkTransaction($payloadDecoded['paymentId'], $this->scopeConfig->getValue('payment/coinaccepted_payment/private_key'), $this->scopeConfig->getValue('payment/coinaccepted_payment/public_key')))) {
            $this->logger->debug(['message' => 'REST response validation error.', 'order_id' => $order->getEntityId()]);

            return $response->setHttpResponseCode(Util::HTTP_CODE_404);
		}

		if ($transactionApi['data']['amountInDestinationCurrency'] !== (float) $order->getGrandTotal()) {
            $this->logger->debug(['message' => 'Invalid price on order.', 'order_id' => $order->getEntityId(), 'expected' => $order->getGrandTotal(), 'received' => $transactionApi['data']['amountInDestinationCurrency']]);

            return $response->setHttpResponseCode(Util::HTTP_CODE_404);
		}

        if (!in_array($transactionApi['data']['status'], Util::AVAILABLE_TRANSACTION_STATUSES, true)) {
            $this->logger->debug(['message' => 'Status validation error.', 'order_id' => $order->getEntityId(), 'expected' => Util::AVAILABLE_TRANSACTION_STATUSES, 'received' => $transactionApi['data']['status']]);

            return $response->setHttpResponseCode(Util::HTTP_CODE_404);
        }

        if (Util::TRANSACTION_STATUS_ACTION_REQUIRED === $transactionApi['data']['status']) {
            $this->actionRequired($order, $payloadDecoded);
        } else {
            $this->captureTransaction($order, $payloadDecoded['paymentId']);
        }

        $this->orderRepository->save($order);

        return $response->setHttpResponseCode(Util::HTTP_CODE_200);
	}

    private function actionRequired(OrderInterface $order, array $payloadDecoded): void
    {
        /** @var Order\Payment $payment */
        if (!$payment = $order->getPayment()) {
            return;
        }

        $transaction = $this->getTransaction($payment, $order, $payloadDecoded['paymentId'], TransactionInterface::TYPE_AUTH);

        $actionRequiredLink = $transactionApi['travelRuleFormUrl'] ?? '';

        $comment = (string) __(Payment::TRAVEL_RULE_INFO, Payment::TRAVEL_RULE_LINK, $actionRequiredLink);
        $payment->addTransactionCommentsToOrder($transaction, $comment);
        $payment->setAdditionalInformation(PaymentInfo::ACTION_REQUIRED_ADDITIONAL_INFORMATION, $actionRequiredLink);

        $payment->save();
        $transaction->save();
    }

	private function captureTransaction(OrderInterface $order, string $paymentId): void
	{
        /** @var Order\Payment $payment */
		if (!$payment = $order->getPayment()) {
			return;
		}

        $transaction = $this->getTransaction($payment, $order, $paymentId, TransactionInterface::TYPE_CAPTURE);

        if ($payment->hasAdditionalInformation(PaymentInfo::ACTION_REQUIRED_ADDITIONAL_INFORMATION)) {
            $payment->unsAdditionalInformation(PaymentInfo::ACTION_REQUIRED_ADDITIONAL_INFORMATION);
        }

        $payment->accept();
        $payment->capture();

		$payment->save();
        $transaction->save();
	}

    public function getTransaction(
        Order\Payment $payment,
        OrderInterface $order,
        string $paymentId,
        string $transactionType
    ): TransactionInterface {
        $transaction = $payment->getAuthorizationTransaction();
        if (!$transaction) {
            $transaction = $this->builderInterface->setPayment($payment)
                ->setOrder($order)
                ->setTransactionId($paymentId)
                ->build($transactionType);

            $transaction->setIsClosed(false);
        }

        return $transaction;
    }
}

